(async function () {
  const els = {
    status: document.getElementById('status'),
    host: document.getElementById('host'),
    lastSync: document.getElementById('lastSync'),
    list: document.getElementById('suggestions'),
    refresh: document.getElementById('refresh'),
    openOptions: document.getElementById('openOptions')
  };

  let currentDomain = '';

  init().catch((error) => {
    setStatus(error.message || 'No se pudo cargar la extensión', true);
  });

  async function init() {
    const tab = await getActiveTab();
    currentDomain = extractDomain(tab?.url || '');
    els.host.textContent = currentDomain || 'Dominio no disponible';

    const status = await sendMessage('gpw:getStatus');
    updateStatus(status);

    els.refresh.addEventListener('click', () => refreshSuggestions(true));
    els.openOptions.addEventListener('click', () => {
      chrome.runtime.openOptionsPage();
    });

    refreshSuggestions(false);
  }

  async function refreshSuggestions(force) {
    if (!currentDomain) {
      setListEmpty('No se pudo detectar el dominio activo.');
      return;
    }
    setListEmpty('Cargando sugerencias…');
    try {
      const data = await sendMessage('gpw:getSuggestions', { domain: currentDomain, force });
      renderList(Array.isArray(data?.items) ? data.items : []);
      if (data?.items?.length) {
        setStatus('Sugerencias listas');
      }
    } catch (error) {
      setStatus(error.message || 'No se pudo obtener sugerencias', true);
      setListEmpty('No hay sugerencias.');
    }
  }

  function renderList(items) {
    els.list.innerHTML = '';
    if (!items.length) {
      setListEmpty('No hay credenciales almacenadas para este dominio.');
      return;
    }

    items.slice(0, 5).forEach((item) => {
      const li = document.createElement('li');
      li.innerHTML = `
        <strong>${sanitize(item.username) || '(sin usuario)'}</strong><br />
        <span class="meta">${sanitize(item.site) || sanitize(item.domain) || 'Sitio sin nombre'}</span>
      `;
      els.list.appendChild(li);
    });
  }

  function setListEmpty(message) {
    els.list.innerHTML = `<li class="empty">${sanitize(message)}</li>`;
  }

  function updateStatus(data) {
    if (!data?.hasToken) {
      setStatus('Inicia sesión en las opciones para empezar', true);
      return;
    }
    setStatus('Sesión iniciada y lista');
    if (data?.lastSync?.timestamp) {
      els.lastSync.textContent = new Date(data.lastSync.timestamp).toLocaleString();
    } else {
      els.lastSync.textContent = 'Nunca';
    }
    if (data?.lastError) {
      setStatus(data.lastError, true);
    }
  }

  function setStatus(message, isError = false) {
    els.status.textContent = message;
    els.status.style.color = isError ? '#f87171' : '#e2e8f0';
  }

  function extractDomain(url) {
    try {
      return new URL(url).hostname;
    } catch {
      return '';
    }
  }

  function getActiveTab() {
    return new Promise((resolve) => {
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        resolve(tabs?.[0]);
      });
    });
  }

  function sanitize(value) {
    if (value == null) {
      return '';
    }
    return String(value)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;');
  }

  function sendMessage(type, payload = {}) {
    return new Promise((resolve, reject) => {
      chrome.runtime.sendMessage({ type, ...payload }, (response) => {
        if (chrome.runtime.lastError) {
          reject(new Error(chrome.runtime.lastError.message));
          return;
        }
        if (!response) {
          reject(new Error('Sin respuesta del servicio'));
          return;
        }
        if (response.ok) {
          resolve(response.data);
        } else {
          reject(new Error(response.error || 'Acción rechazada'));
        }
      });
    });
  }
})();

